﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class LabelParameterVersionRequest : public SSMRequest {
 public:
  AWS_SSM_API LabelParameterVersionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "LabelParameterVersion"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The parameter name on which you want to attach one or more labels.</p>
   * <p>You can't enter the Amazon Resource Name (ARN) for a parameter, only the
   * parameter name itself.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  LabelParameterVersionRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The specific version of the parameter on which you want to attach one or more
   * labels. If no version is specified, the system attaches the label to the latest
   * version.</p>
   */
  inline long long GetParameterVersion() const { return m_parameterVersion; }
  inline bool ParameterVersionHasBeenSet() const { return m_parameterVersionHasBeenSet; }
  inline void SetParameterVersion(long long value) {
    m_parameterVersionHasBeenSet = true;
    m_parameterVersion = value;
  }
  inline LabelParameterVersionRequest& WithParameterVersion(long long value) {
    SetParameterVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more labels to attach to the specified parameter version.</p>
   */
  inline const Aws::Vector<Aws::String>& GetLabels() const { return m_labels; }
  inline bool LabelsHasBeenSet() const { return m_labelsHasBeenSet; }
  template <typename LabelsT = Aws::Vector<Aws::String>>
  void SetLabels(LabelsT&& value) {
    m_labelsHasBeenSet = true;
    m_labels = std::forward<LabelsT>(value);
  }
  template <typename LabelsT = Aws::Vector<Aws::String>>
  LabelParameterVersionRequest& WithLabels(LabelsT&& value) {
    SetLabels(std::forward<LabelsT>(value));
    return *this;
  }
  template <typename LabelsT = Aws::String>
  LabelParameterVersionRequest& AddLabels(LabelsT&& value) {
    m_labelsHasBeenSet = true;
    m_labels.emplace_back(std::forward<LabelsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  long long m_parameterVersion{0};
  bool m_parameterVersionHasBeenSet = false;

  Aws::Vector<Aws::String> m_labels;
  bool m_labelsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
