﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/gamelift/GameLiftRequest.h>
#include <aws/gamelift/GameLift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace GameLift {
namespace Model {

/**
 */
class DeregisterGameServerRequest : public GameLiftRequest {
 public:
  AWS_GAMELIFT_API DeregisterGameServerRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeregisterGameServer"; }

  AWS_GAMELIFT_API Aws::String SerializePayload() const override;

  AWS_GAMELIFT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A unique identifier for the game server group where the game server is
   * running.</p>
   */
  inline const Aws::String& GetGameServerGroupName() const { return m_gameServerGroupName; }
  inline bool GameServerGroupNameHasBeenSet() const { return m_gameServerGroupNameHasBeenSet; }
  template <typename GameServerGroupNameT = Aws::String>
  void SetGameServerGroupName(GameServerGroupNameT&& value) {
    m_gameServerGroupNameHasBeenSet = true;
    m_gameServerGroupName = std::forward<GameServerGroupNameT>(value);
  }
  template <typename GameServerGroupNameT = Aws::String>
  DeregisterGameServerRequest& WithGameServerGroupName(GameServerGroupNameT&& value) {
    SetGameServerGroupName(std::forward<GameServerGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A custom string that uniquely identifies the game server to deregister.</p>
   */
  inline const Aws::String& GetGameServerId() const { return m_gameServerId; }
  inline bool GameServerIdHasBeenSet() const { return m_gameServerIdHasBeenSet; }
  template <typename GameServerIdT = Aws::String>
  void SetGameServerId(GameServerIdT&& value) {
    m_gameServerIdHasBeenSet = true;
    m_gameServerId = std::forward<GameServerIdT>(value);
  }
  template <typename GameServerIdT = Aws::String>
  DeregisterGameServerRequest& WithGameServerId(GameServerIdT&& value) {
    SetGameServerId(std::forward<GameServerIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_gameServerGroupName;

  Aws::String m_gameServerId;
  bool m_gameServerGroupNameHasBeenSet = false;
  bool m_gameServerIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace GameLift
}  // namespace Aws
